/**
* \file: GstreamerVideoPipeline.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: AAUTO
*
* \author: Veeraiyan Chidambaram /RBEI/ECF3/ veeraiyan.chidambaram@in.bosch.com
*          J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#include <adit_logging.h>
#include "GstreamerVideoPipeline.h"

#include <inttypes.h>

using namespace std;

LOG_IMPORT_CONTEXT(aauto_video)

#define getErrAsStr(X) (X == EINVAL) ? "EINVAL" : (X == EPERM) ? "EPERM" : "UNKNOWNERR"

namespace adit { namespace aauto
{
using uspi::SharedDataSender;

const char* GstreamerChannelNameStrings[] =
{
        "(null)",
        "video out",
};

/* PRQA: Lint Message 826: deactivation because casting mechanism of GObject
 * throws the finding */
/*lint -save -e826*/

static GMainLoop* messageBusLoop = nullptr;
static int messageBusRef = 0;
static pthread_mutex_t messageBusMutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_t messageBusThread = 0;
static GMainContext *contextBusLoop = nullptr;

static void* mainLoop(void* inData);

static const char* _videoAppSrc = "aauto_video_src";

GstreamerVideoPipeline::GstreamerVideoPipeline(IAditVideoSinkCallbacks* inCallbacks, void* inSessionContext)
{
    bus = nullptr;
    source = nullptr;
    pipeline = nullptr;
    partial = nullptr;
    eosTimeOutNano = OUT_BUS_STOP_DEFAULT_MILLI * ONE_MILLI_IN_NANO;
    channelName = GstreamerChannelName_Invalid;
    watchId = 0;

    pthread_condattr_t cond_attr;
    pthread_condattr_init(&cond_attr);
    /* set clock attribute of the condition to CLOCK_MONOTONIC */
    pthread_condattr_setclock(&cond_attr, CLOCK_MONOTONIC);
    /* initialize the condition variable referenced by mCondition */
    if (0 != pthread_cond_init(&stopCondition, &cond_attr)) {
        assert(&stopCondition);
    }

    pthread_mutex_init(&stopConditionMutex, NULL);
    stopped = false;
    setThreadPrio = true; // set by default
    threadPrio = 0;
    mCallbacks = inCallbacks;
    
    appsrc = nullptr;
    gstVideoSink = nullptr;
    enoughDataHandlerId = 0;
    firstFrameRenderedHandlerId = 0;
    firstFrameRendered = false;
    firstFrame = true;
    running = false;
    prepareTimeStamp = 0;


    display = nullptr;
    mSessionContext = inSessionContext;

    MessageBusAddRef();
}

GstreamerVideoPipeline::~GstreamerVideoPipeline()
{
    pthread_cond_destroy(&stopCondition);
    pthread_mutex_destroy(&stopConditionMutex);

    MessageBusUnref();
}

bool GstreamerVideoPipeline::GstreamerStart(string videoPipeline, int32_t gwidth, int32_t gheight)
{
    // store timestamp to give warning for late rendering
    prepareTimeStamp = GetHostTicks();
    // set marker for first frame
    firstFrame = true;
    firstFrameRendered = false;

    string partialPipeline = videoPipeline;
    LOGD_DEBUG((aauto_video, "video out pipeline: appsrc ! %s", partialPipeline.c_str()));

    // creates 'pipeline' with 'partial' bin inside
    if (!CreatePipeline("aauto_video_out", partialPipeline,
            GstreamerChannelName_VideoOut))
    {
        // error logged
        return false; /* ========== leaving function ========== */
    }

    // create and configure AppSrc
    if (!CreateAppSrc("appsrc", gwidth, gheight, VIDEO_MAX_QUEUE_SIZE))
    {
        // error logged
        return false; /* ========== leaving function ========== */
    }

    // create VideoSink, register for firstFrameRendered callback
    // and get display reference to the surface
    if (!CreateVideoSink())
    {
        // error logged
        return false; /* ========== leaving function ========== */
    }

    // first set to STATE_READY to capture errors (STATE_READY is synchronous)
    auto ret = gst_element_set_state(GST_ELEMENT(pipeline), GST_STATE_READY);
    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((aauto_video, "gst_element_set_state GST_STATE_READY failed"));
        return false;
    }
    //request  WL-display from gstVideoSink when pipeline is ready.
    g_object_get(G_OBJECT (gstVideoSink), wlHandleName.c_str(), &display, NULL);
    if (!display)
    {
        LOG_ERROR((aauto_video, "%s() Couldn't get WL Display", __FUNCTION__));
        display = nullptr;
    }
    else
    {
        LOG_INFO((aauto_video, "%s() Got Wl Display = %p from GStreamer", __FUNCTION__, display));
    }

    // then set to STATE_PLAYING (is asynchronous)
    ret = gst_element_set_state(GST_ELEMENT(pipeline), GST_STATE_PAUSED);
    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((aauto_video, "gst_element_set_state GST_STATE_PAUSED failed"));
        return false;
    }

    if (!PrepareAndWaitForPushing())
    {
        // error logged
        return false;
    }

    running = true;

    return true;
}


void GstreamerVideoPipeline::GstreamerStop()
{
    bool wasRunning = running;
    running = false;

    /* notify that display becomes invalid */
    SharedDataSender::instance().notifySurfaceDataExpiration(mSessionContext);

    if (appsrc != nullptr)
    {
        if (enoughDataHandlerId > 0)
        {
            g_signal_handler_disconnect(appsrc, enoughDataHandlerId);
            enoughDataHandlerId = 0;
        }

        appsrc = nullptr; // is unref'ed by pipeline
    }

    if (gstVideoSink != nullptr)
    {
        if (firstFrameRenderedHandlerId > 0)
        {
            g_signal_handler_disconnect(gstVideoSink, firstFrameRenderedHandlerId);

            firstFrameRenderedHandlerId = 0;
        }

        gst_object_unref(gstVideoSink);
        gstVideoSink = nullptr;
    }

    // wait (error is logged)
    SendAndWaitForEOS();
    // stop
    StopPipeline();

    if (wasRunning)
        LOGD_DEBUG((aauto_video, "video out stopped"));
}

void GstreamerVideoPipeline::GstreamerPush(uint64_t inTimestamp, uint8_t* inDataPtr, size_t inLen)
{
    
    if (!running || appsrc == nullptr)
    {
        // DO NOT unset first frame marker as nothing will be rendered
        LOG_WARN_RATE(60, (aauto_video, "GstreamerVideoOut is not initialized yet (%u times)",
                LOG_RATE_COUNTER));
    }
    else
    {
        // report when first frame entered (diff from Start)
        if (firstFrame)
        {
            // check time difference start until first render
            uint64_t diffNano = GetTickDifference(prepareTimeStamp, GetHostTicks());
            uint64_t diffMilli = diffNano / (1000 * 1000);
            if (diffMilli > 500)
            {
                LOG_WARN((aauto_video, "video out took more than 500ms (%" PRIu64 "ms) to start rendering!",
                        diffMilli));
            }
            else
                LOGD_DEBUG((aauto_video, "video out took %" PRIu64 "ms to render", diffMilli));
        }
        // unset first frame marker
        firstFrame = false;

        GstBuffer* buffer = gst_buffer_new_and_alloc(inLen);
#if GST_CHECK_VERSION(1,0,0)
        GstMapInfo info;
        if(buffer && gst_buffer_map(buffer, &info, GST_MAP_WRITE))
        {
            if(info.data != NULL)
            {
                memcpy(info.data, inDataPtr, inLen);
                gst_buffer_unmap(buffer, &info); 
            }
            else
            {
                LOG_ERROR((aauto_video, "gst_buffer_map failed"));
                if(mCallbacks != nullptr)
                {
                    mCallbacks->notifyErrorCallback(VIDEO_SINK_RENDER_ERROR);
                }
            }
        }
        else
        {
            LOG_ERROR((aauto_video, "gst_buffer_new_and_alloc failed"));
            if(mCallbacks != nullptr)
            {
                mCallbacks->notifyErrorCallback(VIDEO_SINK_RENDER_ERROR);
            }
        }

        GST_BUFFER_TIMESTAMP(buffer) = GST_CLOCK_TIME_NONE;
#else
        memcpy(GST_BUFFER_DATA(buffer), inDataPtr, inLen);
        GST_BUFFER_TIMESTAMP(buffer) = inTimestamp;
#endif

        int ret = gst_app_src_push_buffer(GST_APP_SRC(appsrc), buffer);
        // takes over buffer, no unref required
        if (ret != 0)
        {
            gst_buffer_unref(buffer); // can't be sure if buffer was unref'ed or not

            // min. interval 60 seconds
            LOG_WARN_RATE(60, (aauto_video, "gst_app_src_push_buffer failed: %d (%u times)", ret,
                    LOG_RATE_COUNTER));
        }
    }
}

void GstreamerVideoPipeline::enoughDataHandler(GstElement* inPipeline, GstreamerVideoPipeline* inMe)
{
    (void)inPipeline;
    (void)inMe;

    // min. interval 60 seconds
    LOG_WARN_RATE(60, (aauto_video, "got enough-data, render queue is full (%u times)",
            LOG_RATE_COUNTER));
}

void GstreamerVideoPipeline::firstFrameRenderedHandler(GstElement* element, GstPad* pad,
        gpointer user_data)
{
    if(user_data != nullptr)
    {
        GstreamerVideoPipeline* tmpPipeline = static_cast<GstreamerVideoPipeline*>(user_data);

        if(tmpPipeline->firstFrameRendered == false)
        {
            LOG_INFO((aauto_video, "video first frame rendered"));
            SharedDataSender::instance().transmitSurfaceData(tmpPipeline->mSessionContext, tmpPipeline->display);

            if (tmpPipeline->mCallbacks != nullptr)
            {
                tmpPipeline->mCallbacks->playbackFirstFrameRenderedCallback();
            }
            else
            {
                LOG_ERROR((aauto_video, "GStreamer callbacks are not set"));
            }
            // no more notification for first-frame rendered
            tmpPipeline->firstFrameRendered = true ;
        }
    }
    else
    {
        LOG_ERROR((aauto_video, "firstFrameRenderedHandler::user_data could not be transferred"));
    }
}

bool GstreamerVideoPipeline::PrepareAndWaitForPushing()
{
    LOG_INFO((aauto_video, "%llu wait for GST_STATE_PAUSED", (unsigned long long)GetHostTicks() % (GST_SECOND * 10)));

    // wait for gst_element_set_state is necessary
    auto ret = gst_element_get_state(GST_ELEMENT(pipeline), NULL, NULL, GST_SECOND);
    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((aauto_video, "gst_element_get_state GST_STATE_PAUSED failed"));
        return false;
    }
    else if (ret == GST_STATE_CHANGE_ASYNC)
    {
        LOG_WARN((aauto_video, "gst_element_get_state GST_STATE_PLAYING timed out"));
    }

    LOG_INFO((aauto_video, "%llu set to GST_STATE_PLAYING", (unsigned long long)GetHostTicks() % (GST_SECOND * 10)));
    // then set to STATE_PLAYING (is asynchronous)
    ret = gst_element_set_state(GST_ELEMENT(pipeline), GST_STATE_PLAYING);
    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((aauto_video, "gst_element_set_state GST_STATE_PLAYING failed"));
        return false;
    }

    return true;
}


bool GstreamerVideoPipeline::CreatePipeline(const std::string& inName, const std::string& inLaunch,
        GstreamerChannelNames inChannelName)
{
    channelName = inChannelName;

    GError* error = nullptr;
    partial = gst_parse_bin_from_description(inLaunch.c_str(), TRUE, &error);
    if (error != nullptr)
    {
        LOG_ERROR((aauto_video, "gst_parse_bin_from_description: %d %s", error->code, error->message));
        g_error_free(error);
        return false; /* ========== leaving function ========== */
    }

    LOGD_VERBOSE((aauto_video, "pipeline created"));

    // create pipeline bin
    pipeline = gst_pipeline_new(inName.c_str());
    if (pipeline == nullptr)
    {
        LOG_ERROR((aauto_video, "gst_pipeline_new failed"));
        return false; /* ========== leaving function ========== */
    }

    // add bus callback
    bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline));
    if (bus != nullptr)
    {
        // create watch for bus callback
        source = gst_bus_create_watch(bus);
        if (source == nullptr)
        {
            LOG_ERROR((aauto_video, "gst_bus_create_watch failed"));
            return false; /* ========== leaving function ========== */
        }
        // set busCallBack for watch source
        g_source_set_callback(source, (GSourceFunc)busCallBack, (void*)this, NULL);
        // add watch source to GMainContext so that it will be executed within that context.
        // will be removed by calling g_source_destroy at StopPipeline.
        watchId = g_source_attach(source, contextBusLoop);
    }
    else
    {
        LOG_ERROR((aauto_video, "gst_pipeline_get_bus failed"));
        return false; /* ========== leaving function ========== */
    }

    // pipeline takes ownership of partial, retain reference
    gst_object_ref(partial);
    if (!gst_bin_add(GST_BIN(pipeline), partial))
    {
        // failed, so release partial
        gst_object_unref(partial);
        LOG_ERROR((aauto_video, "could not add configured part to the pipeline"));
        return false; /* ========== leaving function ========== */
    }

    stopped = false;
    return true;
}

bool GstreamerVideoPipeline::CreateAppSrc(const std::string& inName, int32_t gwidth, int32_t gheight, uint64_t maxBytes)
{
    if (nullptr == (appsrc = gst_element_factory_make(inName.c_str(), _videoAppSrc)))
    {
        LOG_ERROR((aauto_video, "could not create appsrc %s", _videoAppSrc));
        return false; /* ========== leaving function ========== */
    }

    if (!gst_bin_add(GST_BIN(pipeline), appsrc))
    {
        LOG_ERROR((aauto_video, "could not add appsrc element %s", _videoAppSrc));
        return false; /* ========== leaving function ========== */
    }

    if (!gst_element_link(appsrc, partial))
    {
        LOG_ERROR((aauto_video, "could not link appsrc %s", _videoAppSrc));
        return false; /* ========== leaving function ========== */
    }

#if GST_CHECK_VERSION(1,0,0)
    GstCaps* caps = gst_caps_new_simple("video/x-h264",
                                        "stream-format", G_TYPE_STRING, "byte-stream",
                                        NULL);

#else
    GstCaps* caps = gst_caps_new_simple("video/x-h264",
                                        "width", G_TYPE_INT, gwidth,
                                        "height", G_TYPE_INT, gheight,
                                        "framerate", GST_TYPE_FRACTION, 0, 1,
                                        "alignment", G_TYPE_STRING, "au",
                                        "stream-format", G_TYPE_STRING, "byte-stream",
                                        NULL);
#endif

    if (caps != nullptr)
    {
        gst_app_src_set_caps(GST_APP_SRC(appsrc), caps);
        gst_caps_unref(caps);
    }
    else
    {
        LOG_ERROR((aauto_video, "could not set video out caps"));
        return false;
    }

    enoughDataHandlerId = g_signal_connect(appsrc, "enough-data",
            G_CALLBACK (&GstreamerVideoPipeline::enoughDataHandler), this);
    if (enoughDataHandlerId == 0)
        LOG_ERROR((aauto_video, "could not connect not enough-data video out signal"));

    g_object_set(G_OBJECT(appsrc),
            "is-live",      TRUE,               // live source
            "do-timestamp", FALSE,              // don't create timestamps
            "block",        FALSE,              // do not block if queue is full
            "min-percent",  0,                  // always empty queue until 0%
            "max-bytes",    (guint64)maxBytes,  // max queue size
            NULL);
    gst_app_src_set_latency(GST_APP_SRC(appsrc), 0, 100000); // TODO performance: configuration? what is appropriate?

    return true;
}

bool GstreamerVideoPipeline::CreateVideoSink()
{
    GstIterator* iter = gst_bin_iterate_sinks(GST_BIN(partial));
    if (iter == nullptr)
    {
        LOG_ERROR((aauto_video, "Could not get the iterator from GStreamer for the sinks"));
        return false;
    }

#if GST_CHECK_VERSION(1,0,0)
    /* temporary variable */
    GValue gValue = G_VALUE_INIT;
    if(gst_iterator_next(iter, &gValue) == GST_ITERATOR_OK)    //Only one sink will be there
    {
        /* get sink of type GstElement from GValue */
        gstVideoSink = (GstElement*)g_value_get_object(&gValue);
#else
    /*get sink element */
    if(gst_iterator_next(iter, (void**)&gstVideoSink) == GST_ITERATOR_OK)    //Only one sink will be there
    {
#endif
        string frameRenderedSignalName;

        const std::string name = G_OBJECT_TYPE_NAME(G_OBJECT(gstVideoSink));
        if (name.compare("GstApxSink") == 0 )
        {
            wlHandleName = "wl_display";
            frameRenderedSignalName = "first-videoframe-rendered";
        }
        else if((name.compare("GstWaylandSink") == 0 ))
        {
            wlHandleName = "wl-display";
            frameRenderedSignalName = "first-videoframe-rendered";
        }
        else if(name.compare("GstMfxSink") == 0 )
        {
            wlHandleName = "display_ref";
            frameRenderedSignalName = "handoff";
        }
        else
        {
            LOG_ERROR((aauto_video, "Could not find video sink '%s' to register for first-videoframe-rendered signal", name.c_str()));
            gst_iterator_free(iter);
            gst_object_unref(gstVideoSink);
            gstVideoSink = nullptr;
            return false;
        }

        LOG_INFO((aauto_video, "sink used = %s",name.c_str() ));
        //register first frame-buffer rendered signal handler
        firstFrameRenderedHandlerId = g_signal_connect(gstVideoSink, frameRenderedSignalName.c_str(),
                G_CALLBACK (&GstreamerVideoPipeline::firstFrameRenderedHandler), this);

        if (firstFrameRenderedHandlerId == 0)
        {
            LOG_ERROR((aauto_video, "Could not connect to %s signal of Sink %s",frameRenderedSignalName.c_str(), name.c_str()));
            gst_object_unref(gstVideoSink);
            gstVideoSink = nullptr;
            return false;
        }

        gst_iterator_free(iter);
    }
    else
    {
        LOG_ERROR((aauto_video, "GStreamer iterator did not return a valid value as a sink"));
        gst_iterator_free(iter);
        return false;
    }

    return true;
}

bool GstreamerVideoPipeline::SendAndWaitForEOS()
{
    if (pipeline != nullptr && bus != nullptr && watchId != 0)
    {
        uint64_t timeTakenNano = GetHostTicks();
        LOGD_DEBUG((aauto_video, "send EOS to %s", GstreamerChannelNameStrings[channelName]));
        gst_element_send_event(pipeline, gst_event_new_eos());

        // wait for stop condition (EOS or ERROR in bus)
        pthread_mutex_lock(&stopConditionMutex);
        // internally unlocks mutex
        struct timespec timeout;

        if (eosTimeOutNano > 0) {
            /* get the current absolute system time */
            clock_gettime(CLOCK_MONOTONIC, &timeout);
            /* calculate the exceeds time */
            timeout.tv_sec += eosTimeOutNano / ONE_SEC_IN_NANO;
            timeout.tv_nsec += eosTimeOutNano % ONE_SEC_IN_NANO;
            if ((uint64_t)timeout.tv_nsec >= ONE_SEC_IN_NANO) {
                timeout.tv_sec++;
                timeout.tv_nsec -= ONE_SEC_IN_NANO;
            }
        }

        int error = 0;
        while (!stopped)
        {
            error = pthread_cond_timedwait(&stopCondition, &stopConditionMutex, &timeout);
            if (error != 0)
                break;
        }
        pthread_mutex_unlock(&stopConditionMutex);

        timeTakenNano = GetHostTicks() - timeTakenNano;

        if (error == ETIMEDOUT)
        {
            LOG_WARN((aauto_video, "%s timed out while waiting for EOS",
                    GstreamerChannelNameStrings[channelName]));
        }
        else if (error != 0)
        {
            LOG_ERROR((aauto_video, "%s failed to wait for EOS",
                    GstreamerChannelNameStrings[channelName]));
        }
        else
        {
            // success
            LOGD_DEBUG((aauto_video, "%s successfully waited for EOS (after %" PRIu64 "ms)",
                    GstreamerChannelNameStrings[channelName], timeTakenNano / ONE_MILLI_IN_NANO));
        }
    }

    return true;
}

void GstreamerVideoPipeline::StopPipeline()
{
    stopped = true;

    if (pipeline != nullptr)
    {
        auto ret = gst_element_set_state(GST_ELEMENT(pipeline), GST_STATE_NULL);
        if (ret == GST_STATE_CHANGE_FAILURE)
        {
            // silent warning
            LOGD_DEBUG((aauto_video, "gst_element_set_state GST_STATE_NULL failed"));
        }

        gst_element_get_state(GST_ELEMENT(pipeline), NULL, NULL, 1);

        gst_object_unref(pipeline);
        pipeline = nullptr;
    }

    if (watchId != 0)
    {
        watchId = 0;
    }

    if (source != nullptr)
    {
        // removes source from GMainContext and mark it as destroyed
        g_source_destroy(source);
        source = nullptr;
    }

    if (bus != nullptr)
    {
        gst_object_unref(bus);
        bus = nullptr;
    }

    if (partial != nullptr)
    {
        // unreferencing partial releases wl_display, which is shared with input endpoint
        gst_object_unref(partial);
        partial = nullptr;
    }
}

int GstreamerVideoPipeline::SetThreadPriority(const char* threadName, uint32_t priority)
{
    int err;
    struct sched_param schedParam;
    int schedPolicy;

    err = pthread_getschedparam(pthread_self(), &schedPolicy, &schedParam);
    if(err == 0)
    {
        schedParam.sched_priority = priority;
        err = pthread_setschedparam(pthread_self(), SCHED_FIFO, &schedParam);
        if(err != 0)
        LOG_ERROR((aauto_video, "%s: set priority failed with error %s", threadName, getErrAsStr(err)));
    }
    else
    {
        LOG_ERROR((aauto_video, "%s: get priority failed with error %s", threadName, getErrAsStr(err)));
    }

    return err;
}

gboolean busCallBack(GstBus* inBus, GstMessage* inMessage, gpointer inPointer)
{
    (void)inBus;

    auto me = static_cast<GstreamerVideoPipeline*>(inPointer);

    switch (GST_MESSAGE_TYPE(inMessage))
    {
        // GST_EOS: may be ignored verbosely
        case GST_MESSAGE_EOS:
        {
            LOGD_DEBUG((aauto_video, "EOS in %s", GstreamerChannelNameStrings[me->channelName]));

            // signal stop condition
            pthread_mutex_lock(&me->stopConditionMutex);
            me->stopped = true;
            pthread_cond_signal(&me->stopCondition);
            pthread_mutex_unlock(&me->stopConditionMutex);
            break;
        }
        case GST_MESSAGE_ERROR:
        {
            gchar* debug;
            GError* err;
            gst_message_parse_error(inMessage, &err, &debug);
            LOG_ERROR((aauto_video, "Gstreamer error in %s: %s",
                    GstreamerChannelNameStrings[me->channelName],
                    err->message));
            g_error_free(err);
            g_free(debug);

            // don't stop and be silent towards Aauto
            // but signal stop condition
            pthread_mutex_lock(&me->stopConditionMutex);
            me->stopped = true;
            pthread_cond_signal(&me->stopCondition);
            pthread_mutex_unlock(&me->stopConditionMutex);
            break;
        }
        default:
            break;
    }

    return TRUE;
}

gboolean quitCallBack(gpointer data)
{
    (void)data;

    if (messageBusLoop != nullptr)
    {
        // unblock g_main_loop_run
        g_main_loop_quit(messageBusLoop);
        g_main_loop_unref(messageBusLoop);
        messageBusLoop = nullptr;
    }
    else
    {
        LOG_ERROR((aauto_video, "quitCallBack()  GMainLoop is NULL"));
    }

    return G_SOURCE_REMOVE;
}

void MessageBusAddRef()
{
    // one static shared main loop for the message bus
    pthread_mutex_lock(&messageBusMutex);

    messageBusRef++;
    if (messageBusRef == 1)
    {
        if (0 != pthread_create(&messageBusThread, nullptr, mainLoop, nullptr))
        {
            messageBusThread = 0;
            LOG_ERROR((aauto_video, "MessageBusAddRef()  could not create Gstreamer message bus thread"));
        }
    }

    pthread_mutex_unlock(&messageBusMutex);
}

void MessageBusUnref()
{
    // one static shared main loop for the message bus
    pthread_mutex_lock(&messageBusMutex);

    messageBusRef--;
    if (messageBusRef == 0)
    {
        if (messageBusLoop != nullptr)
        {
            // use idle_source to make sure GMainLoop is running
            // when we want to stop GMainLoop from running
            GSource* idle = g_idle_source_new();
            if (idle != nullptr)
            {
                if (contextBusLoop != nullptr)
                {
                    // set callback for idle_source which stops GMainLoop from running
                    g_source_set_callback (idle, (GSourceFunc)quitCallBack, nullptr, nullptr);
                    // add idle_source to GMainContext so that it will be executed within that context
                    g_source_attach(idle, contextBusLoop);
                    g_source_unref(idle);
                }
                else
                {
                    LOG_ERROR((aauto_video, "MessageBusUnref()  GMainContext is NULL"));
                    g_source_destroy(idle);
                    idle = nullptr;
                }
            }
            else
            {
                LOG_ERROR((aauto_video, "MessageBusUnref()  g_idle_source_new failed"));
            }
        }
        else
        {
            LOG_ERROR((aauto_video, "MessageBusUnref()  GMainLoop is NULL"));
        }

        if (messageBusThread != 0)
        {
            pthread_join(messageBusThread, nullptr);
            messageBusThread = 0;
        }
    }

    pthread_mutex_unlock(&messageBusMutex);
}

void* mainLoop(void* inData)
{
    (void)inData;

    // set thread name
    prctl(PR_SET_NAME, "GstMainLoop", 0, 0, 0);

    // create GMainContext
    contextBusLoop = g_main_context_new();
    if (contextBusLoop != nullptr)
    {
        // create GMainLoop
        messageBusLoop = g_main_loop_new(contextBusLoop, FALSE);
        if (messageBusLoop != nullptr)
        {
            g_main_context_push_thread_default(contextBusLoop);
            // this call is blocking until g_main_loop_quit called
            g_main_loop_run(messageBusLoop);
        }
        else
        {
            LOG_ERROR((aauto_video, "mainLoop()  g_main_loop_new failed"));
        }

        // unref context
        g_main_context_pop_thread_default(contextBusLoop);
        g_main_context_unref(contextBusLoop);
        contextBusLoop = nullptr;
    }
    else
    {
        LOG_ERROR((aauto_video, "mainLoop()  g_main_context_new failed"));
    }
    return nullptr;
}

} } // namespace adit { namespace aauto

/*lint -restore*/
